// js/app.js
// Archive Manager – ZIP-only, with ZipCrypto password support and UI improvements.

(() => {
  'use strict';

  const { crc32, deflateRaw, inflateRaw, concatU8 } = window.FFLATE;
  const { encrypt: zipCryptoEncrypt, decrypt: zipCryptoDecrypt } = window.ZipCrypto;

  const $ = (sel) => document.querySelector(sel);

  const fmtBytes = (n) => {
    if (!Number.isFinite(n)) return '—';
    const u = ['B', 'KB', 'MB', 'GB', 'TB'];
    let i = 0;
    let v = n;
    while (v >= 1024 && i < u.length - 1) {
      v /= 1024;
      i++;
    }
    const digits = v >= 100 ? 0 : v >= 10 ? 1 : 2;
    return v.toFixed(digits) + ' ' + u[i];
  };

  const escapeHtml = (s) =>
    String(s).replace(/[&<>"']/g, (c) => ({ '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#39;' }[c]));

  // ---------- Toasts ----------
  const toastHost = $('#toasts');

  function toast(title, body, ms = 3200) {
    const el = document.createElement('div');
    el.className = 'toast';
    el.innerHTML = `<div class="tTitle">${escapeHtml(title)}</div>${
      body ? `<div class="tBody">${escapeHtml(body)}</div>` : ''
    }`;
    toastHost.appendChild(el);
    setTimeout(() => {
      el.style.opacity = '0';
      el.style.transform = 'translateY(6px)';
      el.style.transition = 'all .25s ease';
    }, ms);
    setTimeout(() => el.remove(), ms + 320);
  }

  // ---------- History ----------
  const history = [];

  function addHistory(kind, title, ok, meta) {
    history.unshift({ ts: Date.now(), kind, title, ok, meta });
    if (history.length > 10) history.pop();
    renderHistory();
  }

  function renderHistory() {
    const host = $('#history');
    host.innerHTML = '';
    if (!history.length) {
      const empty = document.createElement('div');
      empty.className = 'small';
      empty.textContent = 'No operations yet';
      host.appendChild(empty);
      return;
    }
    for (const h of history) {
      const dt = new Date(h.ts);
      const el = document.createElement('div');
      el.className = 'hItem';
      el.innerHTML = `
        <div class="hLeft">
          <div class="hTitle">${escapeHtml(h.title)}</div>
          <div class="hMeta">${escapeHtml(h.kind)} • ${escapeHtml(dt.toLocaleString())}${
        h.meta ? ' • ' + escapeHtml(h.meta) : ''
      }</div>
        </div>
        <div class="hTag ${h.ok ? 'ok' : 'bad'}">${h.ok ? 'OK' : 'Error'}</div>
      `;
      host.appendChild(el);
    }
  }

  $('#btnClearHistory').addEventListener('click', () => {
    history.length = 0;
    renderHistory();
    toast('History cleared');
  });

  // ---------- Tabs ----------
  const tabExtract = $('#tabExtract');
  const tabCompress = $('#tabCompress');
  const panelExtract = $('#panelExtract');
  const panelCompress = $('#panelCompress');

  function setTab(which) {
    const isExtract = which === 'extract';
    tabExtract.classList.toggle('active', isExtract);
    tabCompress.classList.toggle('active', !isExtract);
    tabExtract.setAttribute('aria-selected', String(isExtract));
    tabCompress.setAttribute('aria-selected', String(!isExtract));
    panelExtract.classList.toggle('active', isExtract);
    panelCompress.classList.toggle('active', !isExtract);
    panelExtract.hidden = !isExtract;
    panelCompress.hidden = isExtract;
  }

  tabExtract.addEventListener('click', () => setTab('extract'));
  tabCompress.addEventListener('click', () => setTab('compress'));

  // ---------- Drawer ----------
  const drawer = $('#drawer');
  const backdrop = $('#backdrop');

  function openDrawer() {
    drawer.hidden = false;
    backdrop.hidden = false;
    requestAnimationFrame(() => {
      drawer.classList.add('open');
      backdrop.classList.add('open');
    });
  }

  function closeDrawer() {
    drawer.classList.remove('open');
    backdrop.classList.remove('open');
    setTimeout(() => {
      drawer.hidden = true;
      backdrop.hidden = true;
    }, 260);
  }

  $('#btnSettings').addEventListener('click', openDrawer);
  $('#btnCloseDrawer').addEventListener('click', closeDrawer);
  backdrop.addEventListener('click', closeDrawer);

  document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape' && !drawer.hidden && drawer.classList.contains('open')) {
      closeDrawer();
    }
  });

  // ---------- Supported formats dialog ----------
  const dlgFormats = $('#dlgFormats');
  $('#btnFormats').addEventListener('click', () => dlgFormats.showModal());

  // ---------- Operation popup overlay ----------
  const opOverlay = $('#opOverlay');

  function refreshOpOverlayVisibility() {
    const hasExtract = !$('#cardProgressExtract').hidden;
    const hasZip = !$('#cardProgressZip').hidden;
    const shouldShow = hasExtract || hasZip;
    if (shouldShow) {
      opOverlay.hidden = false;
      requestAnimationFrame(() => opOverlay.classList.add('open'));
    } else {
      opOverlay.classList.remove('open');
      setTimeout(() => {
        opOverlay.hidden = true;
      }, 220);
    }
  }

  // ---------- Progress helpers ----------
  function setProgressExtract(show, pct = 0, file = '—', line = '', opTitle) {
    const card = $('#cardProgressExtract');
    const fill = $('#progFillExtract');
    const text = $('#progTextExtract');
    const fname = $('#progFileExtract');
    const hint = $('#opHintExtract');
    const logEl = $('#logExtract');
    const titleEl = $('#opTitleExtract');

    card.hidden = !show;

    if (!show) {
      fill.style.width = '0%';
      text.textContent = '0%';
      fname.textContent = '—';
      hint.textContent = 'Working…';
      logEl.textContent = '';
      titleEl.textContent = 'Extracting…';
      refreshOpOverlayVisibility();
      return;
    }

    if (opTitle) titleEl.textContent = opTitle;

    const p = Math.max(0, Math.min(100, pct | 0));
    fill.style.width = p + '%';
    text.textContent = p + '%';
    fname.textContent = file || '—';
    if (line) {
      logEl.textContent += line + '\n';
      logEl.scrollTop = logEl.scrollHeight;
    }

    refreshOpOverlayVisibility();
  }

  function setProgressZip(show, pct = 0, file = '—', line = '', opTitle) {
    const card = $('#cardProgressZip');
    const fill = $('#progFillZip');
    const text = $('#progTextZip');
    const fname = $('#progFileZip');
    const hint = $('#opHintZip');
    const logEl = $('#logZip');
    const titleEl = $('#opTitleZip');

    card.hidden = !show;

    if (!show) {
      fill.style.width = '0%';
      text.textContent = '0%';
      fname.textContent = '—';
      hint.textContent = 'Working…';
      logEl.textContent = '';
      titleEl.textContent = 'Building ZIP…';
      refreshOpOverlayVisibility();
      return;
    }

    if (opTitle) titleEl.textContent = opTitle;

    const p = Math.max(0, Math.min(100, pct | 0));
    fill.style.width = p + '%';
    text.textContent = p + '%';
    fname.textContent = file || '—';
    if (line) {
      logEl.textContent += line + '\n';
      logEl.scrollTop = logEl.scrollHeight;
    }

    refreshOpOverlayVisibility();
  }

  function onProgress(msg) {
    const { op, pct, file, line, title } = msg;
    if (op === 'extract') setProgressExtract(true, pct, file, line, title);
    if (op === 'zip') setProgressZip(true, pct, file, line, title);
  }

  function downloadBlob(blob, filename) {
    const a = document.createElement('a');
    const url = URL.createObjectURL(blob);
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    a.remove();
    setTimeout(() => URL.revokeObjectURL(url), 1000);
  }

  const supportsFSA = () => !!window.showDirectoryPicker;

  async function writeFileToDir(rootDirHandle, path, dataUint8) {
    const parts = String(path).split('/').filter(Boolean);
    let dir = rootDirHandle;
    for (let i = 0; i < parts.length; i++) {
      const part = parts[i];
      const isLast = i === parts.length - 1;
      if (isLast) {
        const fh = await dir.getFileHandle(part, { create: true });
        const w = await fh.createWritable();
        await w.write(dataUint8);
        await w.close();
      } else {
        dir = await dir.getDirectoryHandle(part, { create: true });
      }
    }
  }

  function sanitizeZipName(name) {
    let n = name.trim();
    if (!n.toLowerCase().endsWith('.zip')) n += '.zip';
    n = n.replace(/[\\/:*?"<>|]+/g, '_');
    if (!n) n = 'archive.zip';
    return n;
  }

  // ---------- Drag helpers ----------
  function wireDrop(zone, onFiles) {
    const prevent = (e) => {
      e.preventDefault();
      e.stopPropagation();
    };
    ['dragenter', 'dragover', 'dragleave', 'drop'].forEach((evt) => zone.addEventListener(evt, prevent));
    zone.addEventListener('dragenter', () => zone.classList.add('drag'));
    zone.addEventListener('dragover', () => zone.classList.add('drag'));
    zone.addEventListener('dragleave', () => zone.classList.remove('drag'));
    zone.addEventListener('drop', (e) => {
      zone.classList.remove('drag');
      const files = Array.from(e.dataTransfer?.files || []);
      onFiles(files);
    });
  }

  // ---------- Extract side ----------
  const fileZip = $('#fileZip');
  const btnPickZip = $('#btnPickZip');
  const btnClearZip = $('#btnClearZip');
  const dropZip = $('#dropZip');

  const pwGate = $('#pwGate');
  const pwGateMessage = $('#pwGateMessage');
  const pwGateForm = $('#pwGateForm');
  const pwInput = $('#pwInput');
  const btnPwUnlock = $('#btnPwUnlock');
  const browserInner = $('#browserInner');

  let openedZipFile = null;
  let openedZipBytes = null;
  let openedZipInfo = null;
  let selected = new Set();
  let cancelTokenExtract = 0;

  let zipRequiresPassword = false;
  let zipHasAES = false;
  let zipPasswordVerified = false;
  let zipPassword = '';

  btnPickZip.addEventListener('click', () => fileZip.click());

  fileZip.addEventListener('change', async () => {
    const f = fileZip.files?.[0];
    if (f) await openZipFile(f);
  });

  btnClearZip.addEventListener('click', clearZip);

  wireDrop(dropZip, async (files) => {
    const f = files.find((x) => x && x.name && x.name.toLowerCase().endsWith('.zip'));
    if (!f) {
      toast('Unsupported file', 'Please drop a .zip file');
      return;
    }
    await openZipFile(f);
  });

  dropZip.addEventListener('keydown', (e) => {
    if (e.key === 'Enter' || e.key === ' ') {
      e.preventDefault();
      fileZip.click();
    }
  });

  function resetPasswordState() {
    zipRequiresPassword = false;
    zipHasAES = false;
    zipPasswordVerified = false;
    zipPassword = '';
    pwInput.value = '';
    pwGate.hidden = true;
    pwGateForm.style.display = '';
    browserInner.hidden = false;
  }

  function clearZip() {
    openedZipFile = null;
    openedZipBytes = null;
    openedZipInfo = null;
    selected.clear();
    resetPasswordState();

    fileZip.value = '';
    $('#zipMeta').hidden = true;
    $('#cardBrowser').hidden = true;
    btnClearZip.disabled = true;
    setProgressExtract(false);
    $('#tbodyExtract').innerHTML = '';
    toast('Cleared');
  }

  async function openZipFile(file) {
    clearZip();
    openedZipFile = file;

    setProgressExtract(true, 0, 'Reading ZIP…', '', 'Opening ZIP…');
    try {
      const ab = await file.arrayBuffer();
      openedZipBytes = new Uint8Array(ab);
      const zip = await parseZip(openedZipBytes);
      openedZipInfo = zip;

      $('#metaName').textContent = file.name;
      $('#metaSize').textContent = fmtBytes(file.size);
      $('#metaEntries').textContent = String(zip.entries.length);

      const anyEncrypted = zip.entries.some((e) => e.encrypted);
      const anyAES = zip.entries.some((e) => e.aes);

      zipRequiresPassword = anyEncrypted;
      zipHasAES = anyAES;
      zipPasswordVerified = !anyEncrypted;
      zipPassword = '';

      let encLabel;
      if (!anyEncrypted) {
        encLabel = 'No password';
      } else if (anyAES) {
        encLabel = 'AES (unsupported – use 7-Zip/WinRAR)';
      } else {
        encLabel = 'ZipCrypto (classic – compatible with Windows Explorer and 7-Zip)';
      }
      $('#metaEncrypted').textContent = encLabel;

      $('#zipMeta').hidden = false;
      btnClearZip.disabled = false;
      $('#cardBrowser').hidden = false;

      if (!anyEncrypted) {
        pwGate.hidden = true;
        browserInner.hidden = false;
        renderExtractTable();
      } else if (anyAES) {
        pwGate.hidden = false;
        browserInner.hidden = true;
        pwGateMessage.textContent =
          'This ZIP uses AES encryption, which is not supported in this app. Please use 7-Zip or WinRAR to open it.';
        pwGateForm.style.display = 'none';
        $('#btnExtractZip').disabled = true;
        $('#btnExtractFolder').disabled = true;
      } else {
        pwGate.hidden = false;
        browserInner.hidden = true;
        pwGateMessage.textContent =
          'This ZIP is password-protected (ZipCrypto). Enter the password to unlock and view the file list.';
        pwGateForm.style.display = '';
        $('#btnExtractZip').disabled = true;
        $('#btnExtractFolder').disabled = true;
        pwInput.focus();
      }

      setProgressExtract(false);
      toast('ZIP opened', `${zip.entries.length} entries`);
      addHistory('Extract', 'Opened ZIP', true, file.name);
    } catch (err) {
      setProgressExtract(false);
      const msg = err?.message || 'Failed to open ZIP';
      toast('Open failed', msg);
      addHistory('Extract', 'Open ZIP failed', false, msg);
    }
  }

  async function unlockZipWithPassword() {
    if (!openedZipInfo || !openedZipBytes) return;
    if (!zipRequiresPassword || zipHasAES) return;

    const password = pwInput.value || '';
    if (!password) {
      toast('Password required', 'Please enter the ZIP password');
      return;
    }

    setProgressExtract(true, 5, 'Verifying password…', '', 'Verifying password…');

    try {
      const probe = openedZipInfo.entries.find((e) => !e.isDir);
      if (probe) {
        await readEntry(openedZipBytes, probe, password);
      }

      zipPassword = password;
      zipPasswordVerified = true;

      pwGate.hidden = true;
      browserInner.hidden = false;
      renderExtractTable();
      refreshSelectionUI();

      setProgressExtract(false);
      toast('ZIP unlocked', 'Password accepted');
      addHistory('Extract', 'ZIP unlocked', true, openedZipFile?.name || '');
    } catch (err) {
      setProgressExtract(false);
      const msg = err?.message || 'Password incorrect';
      toast('Password incorrect', msg);
      addHistory('Extract', 'ZIP unlock failed', false, msg);
    }
  }

  btnPwUnlock.addEventListener('click', unlockZipWithPassword);
  pwInput.addEventListener('keydown', (e) => {
    if (e.key === 'Enter') {
      e.preventDefault();
      unlockZipWithPassword();
    }
  });

  const searchExtract = $('#searchExtract');
  const sortExtract = $('#sortExtract');
  const chkAll = $('#chkAll');

  searchExtract.addEventListener('input', renderExtractTable);
  sortExtract.addEventListener('change', renderExtractTable);

  $('#btnSelectAll').addEventListener('click', () => {
    for (const e of currentExtractList()) selected.add(e.id);
    renderExtractTable();
  });

  $('#btnSelectNone').addEventListener('click', () => {
    selected.clear();
    renderExtractTable();
  });

  chkAll.addEventListener('change', () => {
    const list = currentExtractList();
    if (chkAll.checked) {
      for (const e of list) selected.add(e.id);
    } else {
      for (const e of list) selected.delete(e.id);
    }
    renderExtractTable();
  });

  function currentExtractList() {
    if (!openedZipInfo) return [];
    const q = (searchExtract.value || '').trim().toLowerCase();
    const [field, dir] = (sortExtract.value || 'name:asc').split(':');
    const list = openedZipInfo.entries.filter((e) => !q || e.name.toLowerCase().includes(q));

    const cmp = (a, b) => {
      let va, vb;
      if (field === 'size') {
        va = a.size;
        vb = b.size;
      } else if (field === 'type') {
        va = a.type;
        vb = b.type;
      } else {
        va = a.name;
        vb = b.name;
      }
      if (typeof va === 'string') {
        va = va.toLowerCase();
        vb = vb.toLowerCase();
      }
      if (va < vb) return dir === 'asc' ? -1 : 1;
      if (va > vb) return dir === 'asc' ? 1 : -1;
      return 0;
    };

    list.sort(cmp);
    return list;
  }

  function renderExtractTable() {
    const tbody = $('#tbodyExtract');
    tbody.innerHTML = '';
    if (!openedZipInfo || (!zipPasswordVerified && zipRequiresPassword)) {
      refreshSelectionUI();
      return;
    }

    for (const e of currentExtractList()) {
      const tr = document.createElement('tr');
      const checked = selected.has(e.id);
      tr.innerHTML = `
        <td class="cChk"><input type="checkbox" data-id="${e.id}" ${checked ? 'checked' : ''} /></td>
        <td>${escapeHtml(e.name)}</td>
        <td class="cType">${escapeHtml(e.type)}</td>
        <td class="cSize">${escapeHtml(e.isDir ? '—' : fmtBytes(e.size))}</td>
        <td class="cEnc">${
          e.aes
            ? '<span class="pill bad">AES</span>'
            : e.encrypted
            ? '<span class="pill ok">Yes</span>'
            : '<span class="pill">No</span>'
        }</td>
      `;
      tbody.appendChild(tr);
    }
    tbody.querySelectorAll('input[type="checkbox"][data-id]').forEach((chk) => {
      chk.addEventListener('change', () => {
        const id = Number(chk.getAttribute('data-id'));
        if (chk.checked) selected.add(id);
        else selected.delete(id);
        refreshSelectionUI();
      });
    });
    refreshSelectionUI();
  }

  function refreshSelectionUI() {
    const list = currentExtractList();
    const ids = list.map((e) => e.id);
    const selCount = ids.filter((id) => selected.has(id)).length;
    $('#selSummary').textContent = `${selCount} selected`;

    const all = selCount > 0 && selCount === ids.length;
    chkAll.indeterminate = selCount > 0 && !all;
    chkAll.checked = all;

    const canOperate =
      openedZipInfo &&
      (!zipRequiresPassword || zipPasswordVerified) &&
      !zipHasAES &&
      selCount > 0;

    $('#btnExtractZip').disabled = !canOperate;
    $('#btnExtractFolder').disabled = !canOperate || !supportsFSA();
  }

  $('#btnExtractZip').addEventListener('click', async () => {
    if (!openedZipInfo || !openedZipFile) return;
    const ids = Array.from(selected);
    if (!ids.length) return;

    if (zipHasAES) {
      toast('AES not supported', 'Use 7-Zip or WinRAR to extract AES-encrypted ZIPs');
      return;
    }
    if (zipRequiresPassword && !zipPasswordVerified) {
      toast('Unlock ZIP first', 'Enter the password above to view and extract files');
      return;
    }

    cancelTokenExtract++;
    const token = cancelTokenExtract;

    setProgressExtract(true, 0, 'Preparing…', '', 'Extract: Download as ZIP');
    $('#opHintExtract').textContent = 'Extracting and repacking…';

    try {
      const ab = await openedZipFile.arrayBuffer();
      const files = await extractSelected(new Uint8Array(ab), openedZipInfo, ids, zipPassword || '', token);

      const blobBytes = await createZipFromFiles(
        files.map((f) => ({ path: f.path, data: f.data })),
        { mode: 'store', password: '', cancelToken: token }
      );

      const blob = new Blob([blobBytes], { type: 'application/zip' });
      const name = (openedZipFile.name.replace(/\.zip$/i, '') || 'extracted') + '-extracted.zip';
      downloadBlob(blob, name);

      setProgressExtract(false);
      toast('Done', 'Downloaded extracted ZIP');
      addHistory('Extract', 'Download as ZIP', true, `${files.length} file(s)`);
    } catch (err) {
      setProgressExtract(false);
      const msg = err?.message || 'Extraction failed';
      toast('Extract failed', msg);
      addHistory('Extract', 'Extract failed', false, msg);
    }
  });

  $('#btnExtractFolder').addEventListener('click', async () => {
    if (!supportsFSA()) {
      toast('Not supported', 'Folder extraction is not supported in this browser');
      return;
    }
    if (!openedZipInfo || !openedZipFile) return;

    const ids = Array.from(selected);
    if (!ids.length) return;

    if (zipHasAES) {
      toast('AES not supported', 'Use 7-Zip or WinRAR to extract AES-encrypted ZIPs');
      return;
    }
    if (zipRequiresPassword && !zipPasswordVerified) {
      toast('Unlock ZIP first', 'Enter the password above to view and extract files');
      return;
    }

    cancelTokenExtract++;
    const token = cancelTokenExtract;

    try {
      const dir = await window.showDirectoryPicker({ mode: 'readwrite' });

      setProgressExtract(true, 0, 'Preparing…', '', 'Extract: Extract to Folder');
      $('#opHintExtract').textContent = 'Extracting to folder…';

      const ab = await openedZipFile.arrayBuffer();
      const files = await extractSelected(new Uint8Array(ab), openedZipInfo, ids, zipPassword || '', token);

      let i = 0;
      for (const item of files) {
        if (token !== cancelTokenExtract) throw new Error('Cancelled');
        i++;
        setProgressExtract(
          true,
          Math.floor((i / files.length) * 100),
          item.path,
          `Writing ${i}/${files.length}`,
          'Extract: Extract to Folder'
        );
        await writeFileToDir(dir, item.path, item.data);
      }

      setProgressExtract(false);
      toast('Done', `Extracted ${files.length} file(s)`);
      addHistory('Extract', 'Extract to Folder', true, `${files.length} file(s)`);
    } catch (err) {
      setProgressExtract(false);
      const msg = err?.message || 'Folder extraction failed';
      toast('Extract failed', msg);
      addHistory('Extract', 'Extract to Folder failed', false, msg);
    }
  });

  $('#btnCancelExtract').addEventListener('click', () => {
    cancelTokenExtract++;
    toast('Cancelled', 'Operation cancelled');
    setProgressExtract(false);
  });

  // ---------- Compress side ----------
  let stage = [];
  let stageId = 1;
  let cancelTokenZip = 0;

  const fileAdd = $('#fileAdd');
  const fileAddFolder = $('#fileAddFolder');
  const dropStage = $('#dropStage');

  $('#btnAddFiles').addEventListener('click', () => fileAdd.click());
  $('#btnAddFolder').addEventListener('click', () => fileAddFolder.click());

  fileAdd.addEventListener('change', () => addStageFiles(Array.from(fileAdd.files || []), { fromFolder: false }));
  fileAddFolder.addEventListener('change', () =>
    addStageFiles(Array.from(fileAddFolder.files || []), { fromFolder: true })
  );

  wireDrop(dropStage, (files) => addStageFiles(files, { fromFolder: false }));

  dropStage.addEventListener('keydown', (e) => {
    if (e.key === 'Enter' || e.key === ' ') {
      e.preventDefault();
      fileAdd.click();
    }
  });

  $('#btnClearStage').addEventListener('click', () => {
    stage = [];
    renderStage();
    toast('Staging cleared');
  });

  function guessType(name) {
    const i = name.lastIndexOf('.');
    return i >= 0 ? name.slice(i + 1).toLowerCase() : 'file';
  }

  function addStageFiles(files, { fromFolder }) {
    const added = [];
    for (const f of files) {
      if (!f || typeof f.size !== 'number') continue;
      const path = fromFolder ? f.webkitRelativePath || f.name : f.name;
      const key = `${path}|${f.size}|${f.lastModified}`;
      if (stage.some((x) => `${x.path}|${x.size}|${x.file.lastModified}` === key)) continue;
      const item = {
        id: stageId++,
        file: f,
        path,
        size: f.size,
        type: guessType(path) || (f.type || 'file')
      };
      stage.push(item);
      added.push(item);
    }
    renderStage();
    if (added.length) toast('Added', `${added.length} item(s) staged`);
  }

  function renderStage() {
    const tbody = $('#tbodyStage');
    tbody.innerHTML = '';
    const total = stage.reduce((a, x) => a + x.size, 0);
    $('#stageHint').textContent = `${stage.length} item(s) • ${fmtBytes(total)}`;

    $('#btnClearStage').disabled = stage.length === 0;
    $('#btnBuildZip').disabled = stage.length === 0;

    const list = [...stage].sort((a, b) => a.path.localeCompare(b.path));
    for (const it of list) {
      const tr = document.createElement('tr');
      tr.innerHTML = `
        <td>${escapeHtml(it.path)}</td>
        <td class="cSize">${escapeHtml(fmtBytes(it.size))}</td>
        <td class="cType">${escapeHtml(it.type)}</td>
        <td class="cAction"><button class="btn ghost small" type="button" data-rm="${it.id}">Remove</button></td>
      `;
      tbody.appendChild(tr);
    }
    tbody.querySelectorAll('button[data-rm]').forEach((btn) => {
      btn.addEventListener('click', () => {
        const id = Number(btn.getAttribute('data-rm'));
        stage = stage.filter((x) => x.id !== id);
        renderStage();
      });
    });
  }

  $('#btnBuildZip').addEventListener('click', async () => {
    if (!stage.length) return;

    const name = sanitizeZipName(($('#outName').value || 'archive.zip').trim() || 'archive.zip');
    const mode = $('#cmpLevel').value || 'store';
    const password = $('#outPass').value || '';

    cancelTokenZip++;
    const token = cancelTokenZip;

    setProgressZip(true, 0, 'Preparing…', '', 'Compress: Create ZIP');
    $('#opHintZip').textContent = 'Creating ZIP…';

    try {
      const loaded = [];
      let i = 0;
      for (const it of stage) {
        i++;
        onProgress({
          op: 'zip',
          pct: Math.floor((i / Math.max(1, stage.length)) * 15),
          file: it.path,
          line: `Reading ${i}/${stage.length}`,
          title: 'Compress: Create ZIP'
        });
        const ab = await it.file.arrayBuffer();
        loaded.push({ path: it.path, data: new Uint8Array(ab) });
      }

      const blobBytes = await createZipFromFiles(loaded, { mode, password, cancelToken: token });
      const blob = new Blob([blobBytes], { type: 'application/zip' });
      downloadBlob(blob, name);

      setProgressZip(false);
      toast('ZIP created', 'Download started');
      addHistory('Compress', 'Created ZIP', true, `${stage.length} item(s)`);
    } catch (err) {
      setProgressZip(false);
      const msg = err?.message || 'ZIP creation failed';
      toast('Create failed', msg);
      addHistory('Compress', 'Create ZIP failed', false, msg);
    }
  });

  $('#btnCancelZip').addEventListener('click', () => {
    cancelTokenZip++;
    toast('Cancelled', 'Operation cancelled');
    setProgressZip(false);
  });

  // ---------- Self test ----------
  $('#btnSelfTest').addEventListener('click', async () => {
    const logEl = $('#logSelfTest');
    logEl.textContent = '';
    const log = (s) => {
      logEl.textContent += s + '\n';
      logEl.scrollTop = logEl.scrollHeight;
    };

    try {
      log('1) Creating sample in-memory files');
      const encoder = new TextEncoder();
      const f1 = { path: 'selftest/hello.txt', data: encoder.encode('Hello from Archive Manager\n') };
      const f2 = { path: 'selftest/second.txt', data: encoder.encode('Second file\n') };

      log('2) Creating ZIP (store, no password)');
      const zipBytes = await createZipFromFiles([f1, f2], {
        mode: 'store',
        password: '',
        cancelToken: ++cancelTokenZip
      });
      log(`   ZIP size: ${fmtBytes(zipBytes.byteLength)}`);

      log('3) Opening ZIP');
      const zipInfo = await parseZip(zipBytes);
      log(`   Entries: ${zipInfo.entries.length}`);

      log('4) Extracting and verifying');
      const ids = zipInfo.entries.filter((e) => !e.isDir).map((e) => e.id);
      const files = await extractSelected(zipBytes, zipInfo, ids, '', ++cancelTokenExtract);

      const byPath = new Map(files.map((x) => [x.path, new TextDecoder().decode(x.data)]));
      if (!byPath.get('selftest/hello.txt')?.includes('Hello')) throw new Error('hello.txt mismatch');
      if (!byPath.get('selftest/second.txt')?.includes('Second')) throw new Error('second.txt mismatch');

      log('   Validation OK');
      toast('Self test OK', 'ZIP create/open/extract validated');
      addHistory('SelfTest', 'Self test', true, 'OK');
    } catch (err) {
      const msg = err?.message || String(err);
      log('Self test failed: ' + msg);
      toast('Self test failed', msg);
      addHistory('SelfTest', 'Self test', false, msg);
    }
  });

  // ---------- ZIP parsing / building ----------

  const SIG_EOCD = 0x06054b50;
  const SIG_CEN = 0x02014b50;
  const SIG_LOC = 0x04034b50;

  const u16 = (b, o) => b[o] | (b[o + 1] << 8);
  const u32 = (b, o) => (b[o] | (b[o + 1] << 8) | (b[o + 2] << 16) | (b[o + 3] << 24)) >>> 0;

  function writeU16(arr, n) {
    arr.push(n & 0xff, (n >>> 8) & 0xff);
  }

  function writeU32(arr, n) {
    arr.push(n & 0xff, (n >>> 8) & 0xff, (n >>> 16) & 0xff, (n >>> 24) & 0xff);
  }

  function normalizePath(p) {
    return String(p || '').replace(/\\/g, '/').replace(/^\/+/, '').replace(/\0/g, '');
  }

  function sniffType(name) {
    if (name.endsWith('/')) return 'folder';
    const i = name.lastIndexOf('.');
    return i >= 0 ? name.slice(i + 1).toLowerCase() : 'file';
  }

  function decodeName(bytes, utf8Flag) {
    if (utf8Flag) return new TextDecoder('utf-8', { fatal: false }).decode(bytes);
    const s = new TextDecoder('utf-8', { fatal: false }).decode(bytes);
    if (s.includes(' ')) {
      let out = '';
      for (const b of bytes) out += String.fromCharCode(b);
      return out;
    }
    return s;
  }

  function findEOCD(bytes) {
    const maxBack = Math.min(bytes.length, 0xffff + 22);
    for (let i = bytes.length - 22; i >= bytes.length - maxBack; i--) {
      if (i < 0) break;
      if (u32(bytes, i) === SIG_EOCD) return i;
    }
    return -1;
  }

  function hasAESEncryption(extra) {
    let o = 0;
    while (o + 4 <= extra.length) {
      const id = extra[o] | (extra[o + 1] << 8);
      const sz = extra[o + 2] | (extra[o + 3] << 8);
      o += 4;
      if (o + sz > extra.length) break;
      if (id === 0x9901) return true;
      o += sz;
    }
    return false;
  }

  async function parseZip(bytes) {
    const eocd = findEOCD(bytes);
    if (eocd < 0) throw new Error('Not a ZIP file (EOCD not found)');

    const cdCount = u16(bytes, eocd + 10);
    const cdSize = u32(bytes, eocd + 12);
    const cdOff = u32(bytes, eocd + 16);

    if (cdOff === 0xffffffff || cdSize === 0xffffffff || cdCount === 0xffff) {
      throw new Error('ZIP64 not supported');
    }
    if (cdOff + cdSize > bytes.length) throw new Error('Corrupt ZIP (central directory out of range)');

    const entries = [];
    let o = cdOff;
    for (let idx = 0; idx < cdCount; idx++) {
      if (u32(bytes, o) !== SIG_CEN) throw new Error('Corrupt ZIP (bad central directory header)');

      const flags = u16(bytes, o + 8);
      const method = u16(bytes, o + 10);
      const time = u16(bytes, o + 12);
      const date = u16(bytes, o + 14);
      const crcVal = u32(bytes, o + 16);
      const csize = u32(bytes, o + 20);
      const usize = u32(bytes, o + 24);
      const nlen = u16(bytes, o + 28);
      const xlen = u16(bytes, o + 30);
      const clen = u16(bytes, o + 32);
      const lho = u32(bytes, o + 42);

      const nameBytes = bytes.slice(o + 46, o + 46 + nlen);
      const name = normalizePath(decodeName(nameBytes, (flags & (1 << 11)) !== 0));
      const extra = bytes.slice(o + 46 + nlen, o + 46 + nlen + xlen);

      const encrypted = (flags & 1) !== 0;
      const aes = encrypted && (method === 99 || hasAESEncryption(extra));
      const isDir = name.endsWith('/');

      entries.push({
        id: idx + 1,
        name,
        type: sniffType(name),
        size: usize,
        csize,
        method,
        flags,
        crc: crcVal,
        lho,
        encrypted,
        aes,
        isDir,
        time,
        date
      });

      o = o + 46 + nlen + xlen + clen;
    }
    return { entries };
  }

  function expandFolders(allEntries, selectedEntries) {
    const folders = selectedEntries.filter((e) => e.isDir).map((e) => e.name);
    if (!folders.length) return selectedEntries;
    const out = new Map(selectedEntries.map((e) => [e.id, e]));
    for (const e of allEntries) {
      for (const f of folders) {
        if (e.name.startsWith(f)) out.set(e.id, e);
      }
    }
    return Array.from(out.values());
  }

  async function extractSelected(zipBytes, zip, ids, password, cancelToken) {
    const want = new Set(ids.map((x) => Number(x)));
    const sel = zip.entries.filter((e) => want.has(e.id));
    const expanded = expandFolders(zip.entries, sel);

    const filesOut = [];
    let done = 0;
    const total = Math.max(1, expanded.filter((e) => !e.isDir).length);

    for (const e of expanded) {
      if (e.isDir) continue;
      done++;
      onProgress({
        op: 'extract',
        pct: Math.floor((done / total) * 85),
        file: e.name,
        line: `Extracting ${done}/${total}`,
        title: 'Extracting files…'
      });
      const data = await readEntry(zipBytes, e, password);
      filesOut.push({ path: e.name, data });
      if (cancelToken !== cancelTokenExtract && cancelTokenExtract !== 0) throw new Error('Cancelled');
    }

    onProgress({ op: 'extract', pct: 100, file: 'Done', line: 'Complete', title: 'Extracting files…' });
    return filesOut;
  }

  async function readEntry(zipBytes, e, password) {
    if (e.aes) throw new Error('AES-encrypted ZIP not supported');
    if (u32(zipBytes, e.lho) !== SIG_LOC) throw new Error('Corrupt ZIP (bad local header)');

    const flags = u16(zipBytes, e.lho + 6);
    const method = u16(zipBytes, e.lho + 8);
    const nlen = u16(zipBytes, e.lho + 26);
    const xlen = u16(zipBytes, e.lho + 28);
    const dataStart = e.lho + 30 + nlen + xlen;

    const compSize = e.csize;
    if (dataStart + compSize > zipBytes.length) throw new Error('Corrupt ZIP (entry data out of range)');

    let compData = zipBytes.slice(dataStart, dataStart + compSize);

    if ((flags & 1) !== 0) {
      if (!password) throw new Error('Password required');
      compData = zipCryptoDecrypt(compData, password);
    }

    if (method === 0) {
      const out = compData;
      const c = crc32(out);
      if (c !== e.crc) throw new Error('Wrong password or corrupt ZIP (CRC mismatch)');
      return out;
    }

    if (method === 8) {
      const out = await inflateRaw(compData);
      const c = crc32(out);
      if (c !== e.crc) throw new Error('Wrong password or corrupt ZIP (CRC mismatch)');
      return out;
    }

    throw new Error(`Unsupported compression method: ${method}`);
  }

  function dosDateTime(d = new Date()) {
    const year = d.getFullYear();
    const month = d.getMonth() + 1;
    const day = d.getDate();
    const hour = d.getHours();
    const minute = d.getMinutes();
    const second = Math.floor(d.getSeconds() / 2);

    const dosTime = ((hour & 31) << 11) | ((minute & 63) << 5) | (second & 31);
    const dosDate = (((year - 1980) & 127) << 9) | ((month & 15) << 5) | (day & 31);
    return { dosTime, dosDate };
  }

  async function createZipFromFiles(files, { mode = 'store', password = '', cancelToken } = {}) {
    const now = dosDateTime(new Date());
    const locals = [];
    const centrals = [];
    let offset = 0;

    let i = 0;
    for (const f of files) {
      i++;
      onProgress({
        op: 'zip',
        pct: 15 + Math.floor((i / Math.max(1, files.length)) * 70),
        file: f.path,
        line: `Adding ${i}/${files.length}`,
        title: 'Compress: Create ZIP'
      });

      const name = normalizePath(f.path);
      const nameBytes = new TextEncoder().encode(name);
      const utf8Flag = 1 << 11;

      let method = 0;
      let payload = f.data;

      if (mode === 'deflate') {
        try {
          payload = await deflateRaw(f.data);
          method = 8;
        } catch {
          method = 0;
          payload = f.data;
        }
      }

      const crcVal = crc32(f.data);

      let flags = utf8Flag;
      let compPayload = payload;
      if (password) {
        flags |= 1;
        compPayload = zipCryptoEncrypt(payload, password, crcVal);
      }

      const csize = compPayload.length;
      const usize = f.data.length;

      const lh = [];
      writeU32(lh, SIG_LOC);
      writeU16(lh, 20);
      writeU16(lh, flags);
      writeU16(lh, method);
      writeU16(lh, now.dosTime);
      writeU16(lh, now.dosDate);
      writeU32(lh, crcVal);
      writeU32(lh, csize);
      writeU32(lh, usize);
      writeU16(lh, nameBytes.length);
      writeU16(lh, 0);
      const localHeader = new Uint8Array(lh);
      locals.push(localHeader, nameBytes, compPayload);

      const ch = [];
      writeU32(ch, SIG_CEN);
      writeU16(ch, 0x031e);
      writeU16(ch, 20);
      writeU16(ch, flags);
      writeU16(ch, method);
      writeU16(ch, now.dosTime);
      writeU16(ch, now.dosDate);
      writeU32(ch, crcVal);
      writeU32(ch, csize);
      writeU32(ch, usize);
      writeU16(ch, nameBytes.length);
      writeU16(ch, 0);
      writeU16(ch, 0);
      writeU16(ch, 0);
      writeU16(ch, 0);
      writeU32(ch, 0);
      writeU32(ch, offset);
      const centralHeader = new Uint8Array(ch);
      centrals.push(centralHeader, nameBytes);

      offset += localHeader.length + nameBytes.length + compPayload.length;
      if (offset > 0xffffffff) throw new Error('ZIP64 not supported (archive too large)');
      if (cancelToken && cancelToken !== cancelTokenZip) throw new Error('Cancelled');
    }

    const cdStart = offset;
    const cdBytes = concatU8(centrals);
    const cdSize = cdBytes.length;

    const eocd = [];
    writeU32(eocd, SIG_EOCD);
    writeU16(eocd, 0);
    writeU16(eocd, 0);
    writeU16(eocd, files.length);
    writeU16(eocd, files.length);
    writeU32(eocd, cdSize);
    writeU32(eocd, cdStart);
    writeU16(eocd, 0);
    const eocdBytes = new Uint8Array(eocd);

    onProgress({ op: 'zip', pct: 100, file: 'Done', line: 'Complete', title: 'Compress: Create ZIP' });
    return concatU8([...locals, cdBytes, eocdBytes]);
  }

  // init
  renderHistory();
  renderStage();
})();
