// js/fflate.js
// Minimal ZIP helpers (pure JavaScript, no CompressionStream/DecompressionStream)
//
//   window.FFLATE = {
//     concatU8(arrays),
//     crc32(u8),
//     inflateRaw(u8)   -> Promise<Uint8Array>   // DEFLATE decompression
//     deflateRaw(u8)   -> Promise<Uint8Array>   // stub: throws (caller falls back to store)
//   }

(function (global) {
  'use strict';

  function concatU8(arrs) {
    let len = 0;
    for (const a of arrs) len += a.length;
    const out = new Uint8Array(len);
    let o = 0;
    for (const a of arrs) {
      out.set(a, o);
      o += a.length;
    }
    return out;
  }

  const CRC_TABLE = (() => {
    const t = new Uint32Array(256);
    for (let i = 0; i < 256; i++) {
      let c = i;
      for (let k = 0; k < 8; k++) {
        c = (c & 1) ? (0xEDB88320 ^ (c >>> 1)) : (c >>> 1);
      }
      t[i] = c >>> 0;
    }
    return t;
  })();

  function crc32(u8) {
    let c = 0xFFFFFFFF;
    for (let i = 0; i < u8.length; i++) {
      c = CRC_TABLE[(c ^ u8[i]) & 0xFF] ^ (c >>> 8);
    }
    return (c ^ 0xFFFFFFFF) >>> 0;
  }

  function BitReader(u8) {
    this.u8 = u8;
    this.i = 0;
    this.bit = 0;
    this.buf = 0;
  }

  BitReader.prototype.readBits = function (n) {
    let v = 0;
    for (let k = 0; k < n; k++) {
      if (this.bit === 0) {
        if (this.i >= this.u8.length) throw new Error('Unexpected end of deflate stream');
        this.buf = this.u8[this.i++];
        this.bit = 8;
      }
      const bit = this.buf & 1;
      this.buf >>>= 1;
      this.bit--;
      v |= (bit << k);
    }
    return v;
  };

  BitReader.prototype.alignByte = function () {
    this.bit = 0;
  };

  function reverseBits(code, len) {
    let res = 0;
    for (let i = 0; i < len; i++) {
      res = (res << 1) | (code & 1);
      code >>>= 1;
    }
    return res >>> 0;
  }

  function buildHuffman(lengths) {
    let maxLen = 0;
    for (const l of lengths) if (l > maxLen) maxLen = l;
    if (maxLen === 0) throw new Error('Empty Huffman table');

    const blCount = new Uint16Array(maxLen + 1);
    for (const l of lengths) if (l) blCount[l]++;

    const nextCode = new Uint16Array(maxLen + 1);
    let code = 0;
    for (let bits = 1; bits <= maxLen; bits++) {
      code = (code + blCount[bits - 1]) << 1;
      nextCode[bits] = code;
    }

    const table = [];
    for (let n = 0; n < lengths.length; n++) {
      const len = lengths[n];
      if (!len) continue;
      const raw = nextCode[len]++;
      const rev = reverseBits(raw, len);
      table.push({ code: rev, len, sym: n });
    }
    table.sort((a, b) => (a.len === b.len ? a.code - b.code : a.len - b.len));
    table.maxLen = maxLen;
    return table;
  }

  function decodeSymbol(br, table) {
    let code = 0;
    let len = 0;
    const maxLen = table.maxLen || 15;
    while (len <= maxLen) {
      code |= br.readBits(1) << len;
      len++;
      for (let i = 0; i < table.length; i++) {
        const t = table[i];
        if (t.len === len && t.code === code) return t.sym;
      }
    }
    throw new Error('Invalid Huffman code');
  }

  function buildFixedTables() {
    const litLen = new Uint8Array(288);
    for (let i = 0; i <= 143; i++) litLen[i] = 8;
    for (let i = 144; i <= 255; i++) litLen[i] = 9;
    for (let i = 256; i <= 279; i++) litLen[i] = 7;
    for (let i = 280; i <= 287; i++) litLen[i] = 8;

    const distLen = new Uint8Array(32);
    for (let i = 0; i < 32; i++) distLen[i] = 5;

    return {
      lit: buildHuffman(litLen),
      dist: buildHuffman(distLen)
    };
  }

  const FIXED = buildFixedTables();

  const LEN_BASE  = [3,4,5,6,7,8,9,10,11,13,15,17,19,23,27,31,35,43,51,59,67,83,99,115,131,163,195,227,258];
  const LEN_EXTRA = [0,0,0,0,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,0];
  const DST_BASE  = [1,2,3,4,5,7,9,13,17,25,33,49,65,97,129,193,257,385,513,769,1025,1537,2049,3073,4097,6145,8193,12289,16385,24577];
  const DST_EXTRA = [0,0,0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13];

  function readDynamicTables(br) {
    const HLIT  = br.readBits(5) + 257;
    const HDIST = br.readBits(5) + 1;
    const HCLEN = br.readBits(4) + 4;

    const ORDER = [16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15];
    const clen = new Uint8Array(19);
    for (let i = 0; i < HCLEN; i++) {
      clen[ORDER[i]] = br.readBits(3);
    }
    const clTable = buildHuffman(clen);

    const lengths = [];
    while (lengths.length < HLIT + HDIST) {
      const sym = decodeSymbol(br, clTable);
      if (sym <= 15) {
        lengths.push(sym);
      } else if (sym === 16) {
        const repeat = 3 + br.readBits(2);
        const prev = lengths[lengths.length - 1] || 0;
        for (let i = 0; i < repeat; i++) lengths.push(prev);
      } else if (sym === 17) {
        const repeat = 3 + br.readBits(3);
        for (let i = 0; i < repeat; i++) lengths.push(0);
      } else if (sym === 18) {
        const repeat = 11 + br.readBits(7);
        for (let i = 0; i < repeat; i++) lengths.push(0);
      } else {
        throw new Error('Invalid code-length symbol');
      }
    }

    const litLen  = lengths.slice(0, HLIT);
    const distLen = lengths.slice(HLIT, HLIT + HDIST);
    return { lit: buildHuffman(litLen), dist: buildHuffman(distLen) };
  }

  function inflateRawSync(input) {
    const br = new BitReader(input);
    const out = [];

    let last = false;
    while (!last) {
      last = !!br.readBits(1);
      const type = br.readBits(2);

      if (type === 0) {
        br.alignByte();
        const len  = br.readBits(16);
        const nlen = br.readBits(16);
        if ((len ^ 0xFFFF) !== nlen) throw new Error('Invalid stored block length');
        if (br.i + len > input.length) throw new Error('Stored block exceeds input length');
        for (let i = 0; i < len; i++) out.push(input[br.i++]);
      } else {
        let litTable, distTable;
        if (type === 1) {
          litTable = FIXED.lit;
          distTable = FIXED.dist;
        } else if (type === 2) {
          const dyn = readDynamicTables(br);
          litTable = dyn.lit;
          distTable = dyn.dist;
        } else {
          throw new Error('Reserved deflate block type');
        }

        while (true) {
          const sym = decodeSymbol(br, litTable);
          if (sym < 256) {
            out.push(sym);
          } else if (sym === 256) {
            break;
          } else {
            const lenIndex = sym - 257;
            if (lenIndex < 0 || lenIndex >= LEN_BASE.length) throw new Error('Invalid length code');
            let length = LEN_BASE[lenIndex];
            const eLen = LEN_EXTRA[lenIndex];
            if (eLen) length += br.readBits(eLen);

            const distSym = decodeSymbol(br, distTable);
            if (distSym < 0 || distSym >= DST_BASE.length) throw new Error('Invalid distance code');
            let dist = DST_BASE[distSym];
            const eDist = DST_EXTRA[distSym];
            if (eDist) dist += br.readBits(eDist);

            if (dist > out.length) throw new Error('Invalid distance (backreference out of range)');
            for (let i = 0; i < length; i++) {
              out.push(out[out.length - dist]);
            }
          }
        }
      }
    }

    return Uint8Array.from(out);
  }

  function inflateRaw(u8) {
    return Promise.resolve().then(() => inflateRawSync(u8));
  }

  function deflateRaw(_) {
    return Promise.reject(new Error('DEFLATE compression not implemented; using store mode'));
  }

  global.FFLATE = { concatU8, crc32, inflateRaw, deflateRaw };
})(typeof self !== 'undefined' ? self : window);
